/**
 * \file           ir_driver.cpp
 * \brief          Ir Decoding & Parser/Process Tasks
 * \author         Dennis Hromin
 * \date           09/15/2008
 */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "ir_driver.h"
#include "os.h"
#include "memorymanager.h"
#include "Cresnet.h"
#include "dm_field_debug.h"
#include "console.h"


/*
 Calculations for the serial data over IR mapping ......

 Baud rates to Clock frequency mapping .....
 Supported are 300,600,1200,2400,4800,9600,19200,38400,1800,3600,7200,14400,28800,57600,115200,230400
 The array is in micro seconds and is the length for 1 bit at the specified baud rate.

 bitTimeInMicroSecs = {
        3333.33,     // 300 bps
        1666.66,     // 600 bps
        833.33,      // 1200 bps
        416.66,      // 2400 bps
        208.33,      // 4800 bps
        104.16,      // 9600 bps
        52.08,       // 19200 bps
        26.04,       // 38400 bps
        555.55,      // 1800 bps
        277.77,      // 3600 bps
        138.88,      // 7200 bps
        69.44,       // 14400 bps
        34.72,       // 28800 bps
        17.36,       // 57600 bps
        8.68,        // 115200 bps
        4.34         // 230400 bps
};

So say for 115200, to get 8.68 micro secs, with a clock frequency of 100 nano sec, I do 72 reps of timer1
@ 100 ns + 72 reps of timer 0 @ 20 ns. This gives us nearly a total of 8.64 micro secs...

The numClockCycleRepeats[] is the number of repeats for timer 1 to get 1 bit out at the specified baud rate.
The timer0ClockFrequency[] is the actual frequency for timer 0 and is an integral of the clock cycle. This is
what drives timer 1. Made as small as possible, so that we can get some level of accuracy/granuality.

*/

const UINT16 SerialIrEnvelopeTimingWords[] = {
    66,            // 300 bps  @ 50 micro secs
    33,            // 600 bps  @ 50 micro secs
    1666,          // 1200 bps @ 250 nano secs
    833,           // 2400 bps @ 250 nano secs
    416,           // 4800 bps @ 250 nano secs
    208,           // 9600 bps @ 250 nano secs
    104,           // 19200 bps @ 250 nano secs
    52,            // 38400 bps @ 250 nano secs
    1111,          // 1800 bps @ 250 nano secs
    555,           // 3600 bps @ 250 nano secs
    277,           // 7200 bps  @ 250 nano secs
    139,           // 14400 bps @ 250 nano secs
    71,            // 28800 bps @ 250 nano secs
    35,            // 57600 bps  @ 250 nano secs
    17,            // 115200 bps  @ 250 nano secs
    9,             // 230400 bps  @ 250 nano secs
};

const UINT32 SerialIrBaudrate[] =
{
    300    ,
    600    ,
    1200   ,
    2400   ,
    4800   ,
    9600   ,
    19200  ,
    38400  ,
    1800   ,
    3600   ,
    7200   ,
    14400  ,
    28800  ,
    57600  ,
    115200 ,
    230400
};
/* WARNING: Max size of values in this table is UINT8 (8-bit) */
const UINT32 SerialIrPacing[] =
{
    25,          // 300 bps
    25,          // 600 bps
    25,          // 1200 bps
    25,          // 2400 bps
    25,          // 4800 bps
    50,          // 9600 bps
    100,         // 19200 bps
    100,         // 38400 bps
    25,          // 1800 bps
    25,          // 3600 bps
    25,          // 7200 bps
    100,         // 14400 bps
    100,         // 28800 bps
    100,         // 57600 bps
    100,         // 115200 bps
    200          // 230400 bps
};

/* Table of carrier frequencies to use for each particular baud rate */
/* NOTE: This is the absolute minimum frequency values supported otherwise,
   baud rates will be off and result in high error rates. Min Freq = 4 MHz */
const UINT32 SerialIrCarrierFrequency[] = {

    20000,        // 300 bps    - 20 KHz - 50 micro sec
    20000,        // 600 bps    - 20 KHz - 50 micro sec
    2000000,      // 1200 bps   - 2 MHz - 250 nano sec
    2000000,      // 2400 bps   - 2 MHz - 250 nano sec
    2000000,      // 4800 bps   - 2 MHz - 250 nano sec
    2000000,      // 9600 bps  - 2 Mhz   - 250 nano sec
    2000000,      // 19200 bps - 2 Mhz   - 250 nano sec
    2000000,      // 38400 bps - 2 Mhz   - 250 nano sec
    2000000,      // 1800 bps   - 2 MHz - 250 nano sec
    2000000,      // 3600 bps   - 2 MHz - 250 nano sec
    2000000,      // 7200 bps  - 2 Mhz   - 250 nano sec
    2000000,      // 14400 bps - 2 Mhz   - 250 nano sec
    2000000,      // 28800 bps - 2 Mhz   - 250 nano sec
    2000000,      // 57600 bps - 2 Mhz   - 250 nano sec
    2000000,      // 115200 bps - 2 Mhz   - 250 nano sec
    2000000       // 230400 bps - 2 Mhz   - 250 nano sec
};

/* Table of parity bit mapping for up to 256 bit char */
const UINT8 even_parity[] =
{
0,1,1,0,1,0,0,1,1,0,0,1,0,1,1,0,1,0,0,1,0,1,1,0,0,1,1,0,1,0,0,1,
1,0,0,1,0,1,1,0,0,1,1,0,1,0,0,1,0,1,1,0,1,0,0,1,1,0,0,1,0,1,1,0,
1,0,0,1,0,1,1,0,0,1,1,0,1,0,0,1,0,1,1,0,1,0,0,1,1,0,0,1,0,1,1,0,
0,1,1,0,1,0,0,1,1,0,0,1,0,1,1,0,1,0,0,1,0,1,1,0,0,1,1,0,1,0,0,1,
1,0,0,1,0,1,1,0,0,1,1,0,1,0,0,1,0,1,1,0,1,0,0,1,1,0,0,1,0,1,1,0,
0,1,1,0,1,0,0,1,1,0,0,1,0,1,1,0,1,0,0,1,0,1,1,0,0,1,1,0,1,0,0,1,
0,1,1,0,1,0,0,1,1,0,0,1,0,1,1,0,1,0,0,1,0,1,1,0,0,1,1,0,1,0,0,1,
1,0,0,1,0,1,1,0,0,1,1,0,1,0,0,1,0,1,1,0,1,0,0,1,1,0,0,1,0,1,1,0
};



IrDriver * g_IrDriver;

//void DispIRMsg (UINT8 *pd);


/**
 * \author    Dennis Hromin
 * \brief     Callback function for Ir packet parser main task;  This is a C wrapper
 *            to access the non-static method of the Ir Class; once called,
 *            the method never returns
 * \date      09/15/2008
 * \param     param - anything passed in is ignored by this func
 * \return    void
 * \retval    none
**/
void IrProcessTaskCallback(UINT32 param)
{
  if(g_IrDriver)
        g_IrDriver->IrProcessTask();
}

/**
 * \author        Dennis Hromin
 * \brief         constructor for IrGenerate class
 * \date          09/15/2008
 * \param         void
 * \return        void
 * \retval        none
**/
IrDriver::IrDriver(void):
    m_IrQueue(0),
    m_SerialBytesLeft(0),
    m_bIrSerialStart(0),
    m_MsgAvailable(0),
    m_IrTask(0),
    m_TimingWord(0),
    m_CurrentBit(0),
    m_OutPtr(NULL),
    m_pSerialStrn(0)
{
    UINT16 sStackSize;
    char TaskName[4];

    InitIrDataPacketStruct();

    // determine the stack size; use the greater of the minimum for the processor
    // and the TX minimum
    sStackSize = OsGetDefaultStackVarCnt();
    if (sStackSize < IR_TASK_STACK)
       sStackSize = IR_TASK_STACK;

    // create message queue for Tx task
    m_IrQueue = OsQueueCreate(g_bIRQueueLength,sizeof(UINT32) );

    // create the tasks for the transmitter
    // this is done using a C wrapper funtion for the class since a callback function
    // is required and its difficult to do callbacks in C++ without templates
    sprintf((char *)&TaskName[0], "IR0");
    m_IrTask = OsCreateNamedAdvTask(IrProcessTaskCallback, sStackSize, (UINT32 *)g_IrDriver,
                        IR_TASK_PRIORITY,  (const signed char *)(&TaskName[0]) );
}

/**
 * \author        Andrew Salmon
 * \brief         destructor for IrGenerate class
 * \date          06/11/2009
 * \param         void
 * \return        void
 * \retval        none
**/
IrDriver::~IrDriver()
{
   if(m_IrTask)
   {
      //delete task instance
      OsSuspendAdvTask(m_IrTask);
      OsDeleteAdvTask(m_IrTask);
   }

   if(m_IrQueue)
   {
      //delete the queue
      OsQueueDelete(m_IrQueue);
      m_IrQueue = 0;
   }
}

/**
  * \author      Dennis Hromin
  * \date        09/22/2008
  * \return      None
  * \retval      None
  * \brief       Zeros out the IR Packet structure. It will stop any current transfer.
  * \param       None
  * \note        None
  * \warning     None
  */
void IrDriver::InitIrDataPacketStruct(void)
{
    m_IrPkt.numtimewords = 0;
    m_IrPkt.carrier_freq = 0;
    m_IrPkt.headerlength = 0;
    m_IrPkt.repeatlength = 0;
    m_IrPkt.modulate = 0;
    m_IrPkt.resends = 0;
    m_IrPkt.sendforever = 0;
    m_IrPkt.busy = 0;
    m_IrPkt.transfertype = IR_TRANSFER_CODE;

    memset( m_IrPkt.decoded_irdata_buffer, 0, sizeof( m_IrPkt.decoded_irdata_buffer ));
}


/**
 * \author        Dennis Hromin
 * \brief         Main Ir Processing task that waits on Ir Msg Queue for next packet to be parsed
 * \date          09/15/2008
 * \param         void
 * \return        void
 * \retval        none
**/
void IrDriver::IrProcessTask(void)
{
    UINT8 *pCnetIrPacket = NULL;  //temp pointer to the dequeued message
    UINT8 IRstatus = IR_SUCCESS;

    while ( 1 )
    {
        /* Check if there is already a new mesage pending to be fully processed */
        /* If so, then try to process and just loop back again in some specified delay time to recheck status of queue*/
        if ( !m_MsgAvailable )
        {
            // wait until a message is received (only returns when one is available)
            OsDequeue(m_IrQueue, (void *)&pCnetIrPacket );
            m_MsgAvailable = TRUE;
        }

        /* Otherwise there is a message waiting to be processed, so try to parse it */
        if (pCnetIrPacket)
        {
            if ( !m_IrPkt.busy )
            {
                IRstatus = ParseIrPacket(pCnetIrPacket);
            }
            /* Busy, but previous was a serial IR packet, so continue to decode the rest of current packet */
            else if ( m_IrPkt.transfertype == IR_TRANSFER_SERIAL )
            {
                if ( DecodeIrSerialPacket(pCnetIrPacket) == IR_SERIAL_DECODE_SUCCESS )
                    IRstatus = IR_SUCCESS; //success in decoding, so simply remove irpkt from queue
                else
                    IRstatus = IR_BUSY; //set state to busy.. until the low level driver resets Busy State to false
            }
            /* Busy right now, just wait 10ms and check status flag again  */
            else
                IRstatus = IR_BUSY;
        }

        /* If Current status is anything other than busy, then it was either a success or failure.
                Simply free the memory block now to allow for next to be dequeued */
        if ( IRstatus != IR_BUSY )
        {
            m_MsgAvailable = FALSE;     //we processed the message, so default to no message avail.
            // free the buffer used for this command
            if ( MemMgr && pCnetIrPacket )
                MemMgr->FreeBlock(pCnetIrPacket); //delete the current buffer block
            pCnetIrPacket = NULL;
        }
        /* Else otherwise it must be busy, so loop to top and try again later */
        else
        {
            HwDelayMsec(10);
        }
    }// end while(1) loop
}//end task func

/*
void DispIRMsg (UINT8 *pd)      {

        IR_CONTROL_CREST_DATA_PACKET *IrPacket = (IR_CONTROL_CREST_DATA_PACKET *)pd;
        IR_SERIAL_CREST_DATA_PACKET *IrPacket2;
        UINT8 i;

        if (IrPacket->type != IRSERIAL) {
                DmConsolePrintf ("id=%02x, len=%02x, cmd=%02x, port=%02x, type=%02x, cmdhi=%02x, cmdlow=%02x, dhi=%02x, dlo=%02x\r",
                                                IrPacket->net_id,
                                                IrPacket->len,
                                                IrPacket->cmd,
                                                IrPacket->port,
                                                IrPacket->type,
                                                IrPacket->command_hi,
                                                IrPacket->command_lo,
                                                IrPacket->data_hi,
                                                IrPacket->data_lo);
        }       else    {
                IrPacket2 = (IR_SERIAL_CREST_DATA_PACKET *)pd;

                DmConsolePrintf (" serial id=%02x, len=%02x, cmd=%02x, port=%02x, type=%02x, c_hi=%02x, c_lo=%02x\r",
                                        IrPacket2->net_id,
                                        IrPacket2->len,
                                        IrPacket2->cmd,
                                        IrPacket2->port,
                                        IrPacket2->type,
                                        IrPacket2->comspec_hi,
                                        IrPacket2->comspec_lo);

                DmConsolePrintf ("data ");
                for (i=0; i<(IrPacket2->len-6); i++)    {
                        DmConsolePrintf (" %02x", IrPacket2->data[i]);
                }
                DmConsolePrintf ("\r");
        }
}
*/
/**
 * \author        Dennis Hromin
 * \brief         Main Ir Packet Parsing task that enumerates the command and calls appropriate func to exe
 * \date          09/15/2008
 * \param         pCnetIrPacket - The IR Message packet received via cresnet
 * \return        void
 * \retval        none
**/
UINT8 IrDriver::ParseIrPacket( UINT8 *pCnetIrPacket )
{
    UINT8 IRstatus;

    IR_CONTROL_CREST_DATA_PACKET *IrPacket = (IR_CONTROL_CREST_DATA_PACKET *)pCnetIrPacket;

//    m_IrPkt.source = (UINT8)(IrPacket->net_id); // Store the Source # sent via cresnet (net_id was replaced in packet parser)
    m_IrPkt.port   = (UINT8)(IrPacket->port); // Store the Port # sent via cresnet

    if ( (UINT8)IrPacket->cmd == IR_PACKET ) //IR packet type must be 0x16
    {
        switch ( (UINT8)IrPacket->type )
        {
            case IRSTART:
//                      case IRSTROBE:  // We don't use IRSTROBE, and even if we did - this is not the right place for it
                /* If the HW LowLevel Status returns 0, it is Not Busy, so we can decode the packet */
                if ( IrHwGetLowLevelDriverStatus() == IR_DRIVER_READY )
                {
                    if ( DecodeIrPacket(pCnetIrPacket) == IR_DECODE_SUCCESS )
                    {
                        /* If Decode success, try to start the generate process now */
                        /* Call to Start HW output returns Ready Status if successful, otherwise still busy for some reason */
                        if ( IrHwStartGenerateOutput() == IR_DRIVER_READY )
                            IRstatus = IR_SUCCESS;
                        else /* Otherwise, the lowlevel driver is STILL busy, so we try again later */
                            IRstatus = IR_BUSY;
                    }
                    /* Otherwise, the decoded failed, so return packet error */
                    else
                        IRstatus = IR_INVALID_PACKET;
                }
                /* Otherwise, the lowlevel driver is busy, so we try again later or stop if repeating pulse is still going */
                else
                {
                    /* Driver is still busy, so check if it is currently processing a repeat IR pkt.. then just force a stop */
                    /* NOTE: This check is added mainly for supporting multiplexed IR outputs.. where a stop command is never sent
                            immediately after first start (only at very end). Thus to avoid a deadlock state, we must force a stop here*/
                    if ( m_IrPkt.sendforever == TRUE )
                    {
                        //Removed on 03/07/2012 - This flag modifier is moved down to the driver level now: m_IrPkt.sendforever = FALSE; //Set flag to indicate to driver to stop resending
                                                IrHwForceStopGenerateOutput();
                    }
                    /* Otherwise, still busy.. */
                    IRstatus = IR_BUSY;  //Important to keep status as BUSY so we don't discard this packet from queue.. it has not been decoded yet!
                }
                break;

            case IRSTOP:
                //Removed on 03/07/2012 - This flag modifier is moved down to the driver level now: m_IrPkt.sendforever = FALSE; //Set flag to indicate to driver to stop resending
                IrHwStopGenerateOutput();
                IRstatus = IR_SUCCESS;
                if ( IsDmFieldDebugIndexActive(DM_FIELD_IR_TX_IDX) )
                {
                    DmConsolePrintf( "IR %d: Stop\r", m_IrPkt.port );
                }
                break;

            case IRCONTROL:
                if ( (IrPacket->command_hi == 0) && (IrPacket->command_lo == 0)
                     && (IrPacket->data_hi == 0) && (IrPacket->data_lo == 1) )
                {
                    CresnetSlaveSendIrBusy(m_IrPkt.port, (m_IrPkt.busy)?(0x00):(0x80) ); //pass the port number as the join value
                }
                IRstatus = IR_SUCCESS;
                break;

            case IRSERIAL:
                if ( (IrHwGetLowLevelDriverStatus() != IR_DRIVER_READY) && (m_IrPkt.transfertype == IR_TRANSFER_CODE) )
                {
                    if ( m_IrPkt.sendforever )
                    {
                        //Removed on 03/07/2012 - This flag modifier is moved down to the driver level now: m_IrPkt.sendforever = FALSE; //Set flag to indicate to driver to stop resending
                        IrHwForceStopGenerateOutput(); //force output to go off immediately
                        HwDelayMsec(10);
                        while ( IrHwGetLowLevelDriverStatus() != IR_DRIVER_READY )
                        {
                            HwDelayMsec(10);
                        }
                        m_IrPkt.busy = FALSE;
                    }
                }
                if ( DecodeIrSerialPacket( pCnetIrPacket ) == IR_SERIAL_DECODE_SUCCESS )
                    IRstatus = IR_SUCCESS; //successfuly finsihed decoding and output
                else
                    IRstatus = IR_BUSY; //still busy with decode process
                break;

            default: //Default case.. if there is some serious error in type byte, return with failure
                IRstatus = IR_INVALID_PACKET;
        }
    }
    else
    {
        IRstatus = IR_INVALID_PACKET;
    }

    return IRstatus; //send back the status
}

/**
 * \author      S.Novick
 * \brief       Get a 16-bit word from a big-endian by pointer
 * \date        11/22/2011
 * \retval              UINT16
 * \param       ptr - a pointer to a big-endian argument
 * \note        Not all architectures can deal with misaligned access,
 * \note        so let's read a byte at a time.
 *
**/
UINT16 LoadBE16( UINT8* ptr )
{
    return ((UINT16)*ptr << 8) | *(ptr + 1);
}

/**
 * \author        Dennis Hromin
 * \brief         Ir Decoder Task - waits to see if IR msg in queue, then decode it
 * \date          09/22/2008
 * \retval              BOOL
 * \return      0 - Decode Successful
 * \return      1 - Decode Failed
 * \brief       Reads an IR packet structure and sets up the IroSPI Slave & DMA
 * \param       data - Pointer at a char buffer containing an IR packet structure.
 * \note        None
 * \warning     Transmission begins right away.
 *
**/
BOOL IrDriver::DecodeIrPacket(UINT8 *pCnetIrPacket)
{
    unsigned i;
        IRPACKETHDR *pIrCmdHeader;
    UINT16 *pTimeword;
    UINT16 *pOut;
    UINT8  *pIndexword;

    InitIrDataPacketStruct(); //reset all variables to default state on new data to output

        // Cnet IR packet data[] payload starts with IRPACKETHDR
    pIrCmdHeader = (IRPACKETHDR*)(((IR_START_CREST_DATA_PACKET*)pCnetIrPacket)->data);

    m_IrPkt.transfertype = IR_TRANSFER_CODE;
        // Both header and repeater codes are stored as nibbles, so we must double these values for correct length
        m_IrPkt.headerlength = (pIrCmdHeader->header_length)*2;
        m_IrPkt.repeatlength = (pIrCmdHeader->repeat_length)*2;
    //The number of time words is the lower nibble of byte 5
        m_IrPkt.numtimewords = pIrCmdHeader->number_of_timewords & 0x0F;
    //The number of repeats is the upper nibble of byte 5
        m_IrPkt.resends = (pIrCmdHeader->number_of_timewords & 0xF0) >> 4;
    //A value of 0 for min num repeats maps to 4 repeats by default
    if(m_IrPkt.resends == 0)
        m_IrPkt.resends = 4;
        if(m_IrPkt.repeatlength == 0)
            m_IrPkt.sendforever = FALSE; //This is a strobe IR pulse
        else
                m_IrPkt.sendforever = TRUE; //This is a repeat IR pulse

        /*--- Determine the Carrier Frequencey ---*/
    m_IrPkt.modulate = IR_MODULATE_ENABLED; //Enable Modulation of Carrier by default
    if (pIrCmdHeader->carrier < 48)
        m_IrPkt.modulate = IR_MODULATE_DISABLED; //Disable carrier modulation (i.e. envelope only! )

    switch (pIrCmdHeader->carrier) // carrier
    {
        case 4:
            m_IrPkt.carrier_freq = CARRIER_4; //1.1 MHz
            m_IrPkt.modulate = IR_MODULATE_ENABLED; //Turn carrier back on
            break;
        case 5:
            m_IrPkt.carrier_freq = CARRIER_5; //1 MHz
            m_IrPkt.modulate = IR_MODULATE_ENABLED; //Turn carrier back on
            break;
        case 9:
            m_IrPkt.carrier_freq = CARRIER_9; //455 KHz
            m_IrPkt.modulate = IR_MODULATE_ENABLED; //Turn carrier back on
            break;
        case 11:
            m_IrPkt.carrier_freq = CARRIER_11; //366 KHz
            m_IrPkt.modulate = IR_MODULATE_ENABLED; //Turn carrier back on
            break;
        case 12:
            m_IrPkt.carrier_freq = CARRIER_12; //336 KHz
            m_IrPkt.modulate = IR_MODULATE_ENABLED; //Turn carrier back on
            break;
        default:
            m_IrPkt.carrier_freq = (UINT32)4000000/pIrCmdHeader->carrier;
            break;
    }

    if ( IsDmFieldDebugIndexActive(DM_FIELD_IR_TX_IDX) )
    {
        DmConsolePrintf( "IR %d: HDR length %d, RPT length %d, MIN repeats %d, Carrier %d KHz\r",
            m_IrPkt.port,
            m_IrPkt.headerlength,
            m_IrPkt.repeatlength,
            m_IrPkt.resends,
            m_IrPkt.carrier_freq / 1000 );
    }

        /* Important Memory boundry check: If recieved IR packet has more data than we can decode in current
           buffer setting, then simply return a failure and discard the packet */
        if ((m_IrPkt.headerlength+m_IrPkt.repeatlength) > IR_MAX_DECODE_BUFFER )
                return IR_DECODE_FAIL;  //Return 1 as a Failure Status

        //Decode the Header Data and store into RAM buffer
    pTimeword = pIrCmdHeader->timeword;
    pIndexword = (UINT8*)pTimeword + (m_IrPkt.numtimewords * 2);
    pOut = m_IrPkt.decoded_irdata_buffer;

    for ( i = 0; i < (m_IrPkt.headerlength + m_IrPkt.repeatlength); i += 2 )
    {
        *pOut++ = LoadBE16( (UINT8*)&pTimeword[*pIndexword >> 4] );
        *pOut++ = LoadBE16( (UINT8*)&pTimeword[*pIndexword & 0x0f] );
        pIndexword++;
    }

        /* After the full decode process.. indicate that we are now busy processing the data via the low level driver */
    m_IrPkt.busy = TRUE;
        return IR_DECODE_SUCCESS; //Decoding was successful
}

/**
  * \author      S.Novick
  * \date        11/23/2011
  * \return      None
  * \brief       Adds a serial bit to the output stream of timing words
  * \param       BOOL serial bit value
  * \param       UINT16 Timeword, carrier clocks per bit time
  * \note        None
  * \warning     None
  */
void IrDriver::AddSerialBit( BOOL bit_value, UINT16 Timeword )
{
    if ( bit_value != m_CurrentBit )
    {
        *m_OutPtr++ = m_TimingWord;  //store the final summed up timing words into ptr array
        m_TimingWord = 0;  //reset back to zero and start summing up for next timeword in stream
        m_CurrentBit = bit_value; //store the current new bit (1 or 0) to process
    }
    m_TimingWord += Timeword; //just add in the current timeword param value to the main summing variable
}

/**
  * \author      Dennis Hromin
  * \date        10/20/2008
  * \return      None
  * \retval      0 - success
  * \brief       Decodes a Serial IR Packet and send out data stream
  * \param       pCnetIrPacket - char * to the cresnet packet
  * \note        None
  * \warning     None
  */
BOOL IrDriver::DecodeIrSerialPacket(UINT8* pCnetIrPacket)
{
    UINT8 NumBytesToSend;
    UINT8 baudrate_ArrayIdx;
    UINT16 SerialIrEnvelopeTimeword; //must be 16-bit variable to represent timewords from array
    UINT8 comspeclo;
    UINT8 serial_NumDataBits, serial_NumStopBits, serial_Parity, NumPacingSpaceBits;

    IR_SERIAL_CREST_DATA_PACKET *IrPacket = (IR_SERIAL_CREST_DATA_PACKET *)pCnetIrPacket;

    /*--- If this is the first byte stream, execute these routines for initial setup ---*/
    if(m_IrPkt.busy == FALSE) //if not busy, then this is the first parsing of packet
    {
        InitIrDataPacketStruct(); //reset all variables to default state on new data to output

        m_bIrSerialStart = TRUE; //Start of serial transfer

//        m_IrPkt.source = IrPacket->net_id;  // Store the Source # sent via cresnet (net_id was replaced in packet parser)
        m_IrPkt.port   = IrPacket->port;    // Store the Port # sent via cresnet

        m_IrPkt.transfertype = IR_TRANSFER_SERIAL;
        m_SerialBytesLeft = IrPacket->len - 6; // subtract the size of the serial header from total length
        m_pSerialStrn = IrPacket->data;

        m_IrPkt.sendforever = FALSE;                    // Not a repeating IR pulse
        m_IrPkt.resends = 0;                        // There are no resends
        m_IrPkt.modulate = IR_MODULATE_DISABLED;    // Disable carrier modulation (i.e. envelope only! )

        m_IrPkt.busy = TRUE; //We are now busy... finally
    }  //end of if statement

    /*--- Otherwise, we already sent out some serial string(s), so just continue using same params as before ---*/
    else
    {
        /* Check first if driver is still busy.. do nothing and return when free to further decode or end process */
        if (IrHwGetLowLevelDriverStatus())
        {
                m_IrPkt.busy = TRUE; //We are still busy..
                return IR_SERIAL_DECODE_BUSY;
        }
        else if (m_SerialBytesLeft == 0)
        {
            m_bIrSerialStart = FALSE;

            /* If we sent out all bytes already, check to see if we can properly terminate and set to NOT Busy */
            m_IrPkt.busy = FALSE; //No longer busy..can process other packets
            return IR_SERIAL_DECODE_SUCCESS;
        }
    } //end of else statement

    // If there are more bytes to send than max per cycle - Check how many do we have to send in this cycle
    // if more then MAX_SERIAL_IR_BYTES_SENT_IN_ONE_CYCLE, we send only MAX_SERIAL_IR_BYTES_SENT_IN_ONE_CYCLE
    // else we send the remainder
    NumBytesToSend = min( m_SerialBytesLeft, IR_MAX_SERIAL_BYTES_SENT_IN_ONE_CYCLE );
    m_SerialBytesLeft -= NumBytesToSend;

    /* Parse the Common COM Specs header information to get the partity settings and baudrate..etc. */
    //COM Specs are encoded in byte 7
    comspeclo = IrPacket->comspec_lo;

    // Grab the array Index for the baud rate
    baudrate_ArrayIdx  = ((comspeclo & 0x80) >> 4) | (comspeclo & 0x07) ; // index into baud coeff struct

    // Convert it into how many reps using the System Clock frequency....
    SerialIrEnvelopeTimeword = SerialIrEnvelopeTimingWords[baudrate_ArrayIdx];

    // pacing between characters
    NumPacingSpaceBits = SerialIrPacing[baudrate_ArrayIdx];

    m_IrPkt.carrier_freq = SerialIrCarrierFrequency[baudrate_ArrayIdx]; //select the appropriate carrier freq based on baud rate index into table

    // Get number of data bits and stop bits
    serial_NumDataBits = (comspeclo & 0x20) ? 8 : 7;
    serial_NumStopBits = (comspeclo & 0x40) ? 2 : 1;

    // Grab the parity
    switch(comspeclo & 0x18)
    {
        case 0x00:
            serial_Parity = IR_PARITY_NONE;
            break;
        case 0x08:
            serial_Parity = IR_PARITY_EVEN;
            break;
        case 0x10:
            serial_Parity = IR_PARITY_Z;
            break;
        case 0x18:
            serial_Parity = IR_PARITY_ODD;
            break;
    }

    if ( IsDmFieldDebugIndexActive(DM_FIELD_IR_TX_IDX) )
    {
        size_t string_length = NumBytesToSend + 32;             // ~32 for the header, the rest for the payload
        char *fdebug_string = (char*)MemMgr->GetBlock( string_length );

        if ( fdebug_string != NULL )
        {
            memset( fdebug_string, 0, string_length );
            sprintf( fdebug_string, "IR %d: Serial %d,%d,%c,%d: ", m_IrPkt.port, SerialIrBaudrate[baudrate_ArrayIdx], serial_NumDataBits, serial_Parity, serial_NumStopBits );
            memcpy( fdebug_string + strlen(fdebug_string), m_pSerialStrn, NumBytesToSend );
            fdebug_string[strlen(fdebug_string)] = '\r';
            DmConsolePrintf( fdebug_string );
            MemMgr->FreeBlock( (UINT8*)fdebug_string );
        }
    }
/* Start of common area for decoding the serial chars into bits/timing words */
/*--- Notes about decoding procedure -- Ported from Binky based control system FW ---*/
    // The serial data is of the format
    // START_BIT | DATA_BITS | PARITY_BIT | STOP_BITS
    // The START_BIT is always 0
    // The DATA_BITS can be between 5 to 8 bits
    // PARITY_BIT can be present or not
    // STOP_BITS can be 1 or 2 bits
    // Thus the length can be between 7 bits (START + 5 DATA + NO PARITY + 1 STOP ) to 12 bits
    //  (START + 8 DATA + 1 PARITY + 2 STOPS)
    // Serial will be ENVELOPE only - there will be no "FILL IN MODULATION"
    // At the end of each serial byte we send out XX pause bits - Pause bits have a value of 1.
    // We calculate the number of sequential 0's and 1's and then create a waveform. Each bit is a multiple of
    // the envelopeFrequency. The start bit is always 0 while the stop bits and the pause bits are always 1.

        /* Important: ALWAYS clear out the decode buffer before populating it.. this is a special case when decoding serial IR pkts  */
    memset( m_IrPkt.decoded_irdata_buffer, 0, sizeof( m_IrPkt.decoded_irdata_buffer ));
    ResetSerial();
    m_OutPtr = m_IrPkt.decoded_irdata_buffer;

    /* Added On Jan 24, 2013 by DHROMIN/SNOVICK -- Fixes bug #61484 where 1st char on 115k baud rate would get corrupted, due to bad start bit pulse width. */
    /* Updated On April 25, 2013 by DHROMIN -- Increases 30% to 40% -- Fixes bug #61484 where 1st char on 115k baud rate would get corrupted, due to bad start bit pulse width. */
	/* Fix was made by slightly increasing the 1st timeword pulse generated by ~40% more. Only performed on 115K as other baud rates work fine */
    if( SerialIrBaudrate[baudrate_ArrayIdx] == 115200 )
    {
        AddSerialBit( 0, ((4*SerialIrEnvelopeTimeword)/10) );    // start bit - add additional 40% or ~X ticks to env time word for 115K special case baud rate
    }

    while ( NumBytesToSend-- )
    {
        unsigned i;
        UINT8 dataByteToSend = *m_pSerialStrn;

        AddSerialBit( 0, SerialIrEnvelopeTimeword );    // start bit

        for ( i = 0; i < serial_NumDataBits; i++ )
        {
            AddSerialBit( dataByteToSend & 0x01, SerialIrEnvelopeTimeword );
            dataByteToSend >>= 1;
        }

        if ( serial_Parity != IR_PARITY_NONE )
        {
            UINT8 parity_bit;
            switch (serial_Parity)
            {
                case IR_PARITY_EVEN:
                    if (even_parity[*m_pSerialStrn])
                         parity_bit = 1;
                    else
                         parity_bit = 0;
                    break;
                case (IR_PARITY_Z):
                    parity_bit = 1;  // Mark  - parity bit is always 1
                    break;
                case (IR_PARITY_ODD):
                    if (even_parity[*m_pSerialStrn])
                        parity_bit = 0;
                    else
                        parity_bit = 1;
                    break;
            }
            AddSerialBit( parity_bit, SerialIrEnvelopeTimeword );   // add parity
        }

        AddSerialBit( 1, SerialIrEnvelopeTimeword );        // the first stop bit
        if ( serial_NumStopBits > 1 )
        {
            AddSerialBit( 1, SerialIrEnvelopeTimeword );    // the second, if needed
        }

        for ( i = 0; i < NumPacingSpaceBits; i++ )
        {
            AddSerialBit( 1, SerialIrEnvelopeTimeword );    // add fixed pacing
        }

        m_pSerialStrn++;
    }

    AddSerialBit( 0, SerialIrEnvelopeTimeword );    // Bugzilla# 61461, this just forces out the previous bit.

    // set header lenght to the number of items we stored in the decode_buffer
    m_IrPkt.headerlength = (UINT16)( m_OutPtr - m_IrPkt.decoded_irdata_buffer ); // pointer math evaluates to the number of items
        m_IrPkt.repeatlength = 0; //NO repeats.. so make sure it is set to zero

        /* If there is data to send based on size of header, then start the generate process */
        if(m_IrPkt.headerlength > 0)
          IrHwStartGenerateOutput();

        return IR_SERIAL_DECODE_BUSY;
}



/**
  * \author      Dennis Hromin
  * \date        May 18 2010
  * \return      UINT8
  * \retval      0 - IR_CODETYPE_PULSE
  * \retval      1 - IR_CODETYPE_REPEATING
  * \retval      2 - IR_CODETYPE_SERIAL
  * \brief       Determines the IR Code Type based on current status of packet being decoded. Can be only ONE of 3 IR pkt types
  * \param       void
  * \note        None
  * \warning     None
  */
UINT8 IrDriver::GetIRCodeType(void)
{
        /* For Serail IR Packets Decoded */
        if(m_IrPkt.transfertype == IR_TRANSFER_SERIAL)
                return IR_CODETYPE_SERIAL;
        /* For Normal IR Pulses with Repeating Code Present */
        else if(m_IrPkt.repeatlength != 0x00)
                return IR_CODETYPE_REPEATING;
        /* For Normal IR Pulses with Header only Code Present */
        else
                return IR_CODETYPE_PULSE;

}
